/*
Slit door timing box firmware designed by William Collier for Samsung Austin Semiconductor
When properly connected to the slit door sensors on an AMAT Producer this software will show the timing difference between
the open and closing time of both doors as indicated by the slit door sensors. This will also provide near real time visual 
indication of individaul slit door sensors for troubleshooting purposes.
Slit door inputs are active low so the input goes to 0 when the sensor is made
REV 1.0 - This is the initial version of the software
Rev 2.0 - implamented a debounce function instead of debouncing in the main loop. Fixed the diff timers so they display as base 10 now instead of binary
*/


#include <Wire.h>                          //library for I2C communication
#include <hd44780.h>                       // library for main hd44780 header
#include <hd44780ioClass/hd44780_I2Cexp.h> // library for i2c expander i/o class header

hd44780_I2Cexp lcd;                        // declare lcd object: auto locate & auto config expander chip

//assign digital I/O's
const int s1_OpenOutput=3;
const int s1_CloseOutput=4;
const int s2_OpenOutput=5;
const int s2_CloseOutput=6;
const int s1_OpenInput=7;
const int s1_CloseInput=8;
const int s2_OpenInput=9;
const int s2_CloseInput=10;
const int s1_OpenLED=11;
const int s1_CloseLED=12;
const int s2_OpenLED=13;
const int s2_CloseLED=2;

//setup and assign position variables
int s1_Open=0;
int s1_Close=0;
int s2_Open=0;
int s2_Close=0;
int lastS1_Open=0;
int lastS1_Close=0;
int lastS2_Open=0;
int lastS2_Close=0;
int s1_OpenReading = 0;
int s1_CloseReading = 0;
int s2_OpenReading = 0;
int s2_CloseReading = 0;

//Slit door timing variables
int openTimerStart=0;
int closeTimerStart=0;
int fromClose=0;
int fromOpen=0;
unsigned long openTimer=millis();
float openDiffTime=0;
unsigned long closeTimer=millis();
float closeDiffTime=0;
float openDiffTimeSec=0;
float closeDiffTimeSec=0;

//setup and assign debouncing timing variables
const unsigned long debounceDelay = 20;
unsigned long timerLastStateS1_Open = 0;
unsigned long timerLastStateS1_Close = 0;
unsigned long timerLastStateS2_Open = 0;
unsigned long timerLastStateS2_Close = 0;
//unsigned long timerOpenDifference = 0;
//unsigned long timerCloseDifference = 0;

char s1_Status;
char s2_Status;

const int LCD_COLS=16;                      // LCD columns
const int LCD_ROWS=2;                       // LCD Rows


void setup() 
{




Serial.begin(9600); //Initialis serial output monitoring
Wire.begin(); //Starts I2C serial communication
Wire.setClock(100000);  //Sets the communication clock to standard frequency of 100KHz

pinMode(s1_OpenOutput,OUTPUT);
pinMode(s1_CloseOutput,OUTPUT);
pinMode(s2_OpenOutput,OUTPUT);
pinMode(s2_CloseOutput,OUTPUT);
pinMode(s1_OpenInput,INPUT_PULLUP);
pinMode(s1_CloseInput,INPUT_PULLUP);
pinMode(s2_OpenInput,INPUT_PULLUP);
pinMode(s2_CloseInput,INPUT_PULLUP);
pinMode(s1_OpenLED, OUTPUT);
pinMode(s1_CloseLED, OUTPUT);
pinMode(s2_OpenLED, OUTPUT);
pinMode(s2_CloseLED, OUTPUT);
//read initial valve position sensor status
digitalWrite(s1_OpenOutput, digitalRead(s1_OpenInput));
digitalWrite(s1_CloseOutput, digitalRead(s1_CloseInput));
digitalWrite(s2_OpenOutput, digitalRead(s2_OpenInput));
digitalWrite(s2_CloseOutput, digitalRead(s2_CloseInput));

int status;
  status = lcd.begin(LCD_COLS, LCD_ROWS);
  if(status)                             // non zero status means it was unsuccesful
  {
    // hd44780 has a fatalError() routine that blinks an led if possible
    // begin() failed so blink error code using the onboard LED if possible
    hd44780::fatalError(status); // does not return
  }

//Read the inputs 
s1_OpenReading = digitalRead(s1_OpenInput);
s1_CloseReading = digitalRead(s1_CloseInput);
s2_OpenReading = digitalRead(s2_OpenInput);
s2_CloseReading = digitalRead(s2_CloseInput);

//Test indication LEDs at powerup
digitalWrite (s1_OpenLED, LOW);
digitalWrite (s1_CloseLED, LOW);
digitalWrite (s2_OpenLED, LOW);
digitalWrite (s2_CloseLED, LOW);
delay(2000);
digitalWrite (s1_OpenLED, HIGH);
digitalWrite (s1_CloseLED, HIGH);
digitalWrite (s2_OpenLED, HIGH);
digitalWrite (s2_CloseLED, HIGH);

//Inital slit door position setup
  if (s1_OpenReading==0 && s2_OpenReading==0){
    fromOpen=1;
  }
  if (s1_CloseReading==0 && s2_CloseReading==0){
    fromClose=1;
  }
lcd.clear();

}

void loop() 
{
Serial.print("Input:");
Serial.print(s1_Open);
Serial.print(s1_Close);
Serial.print(s2_Open);
Serial.print(s2_Close);
Serial.print("  ");
Serial.print("Output:");
Serial.print(s1_Open);
Serial.print(s1_Close);
Serial.print(s2_Open);
Serial.print(s2_Close);
Serial.print("  ");
Serial.print("fromClose:");
Serial.print(fromClose);
Serial.print(" fromOpen:");
Serial.print(fromOpen);
Serial.print("  ");
Serial.print("openTimerStart:");
Serial.print(openTimerStart);
Serial.print(" openTimer:");
Serial.print(openTimer);
Serial.print(" openDiffTime:");
Serial.print(openDiffTime);
Serial.print(" openDiffTimeSec:");
Serial.print(openDiffTimeSec);
Serial.print("  ");
Serial.print("closeTimerStart:");
Serial.print(closeTimerStart);
Serial.print(" closeTimer:");
Serial.print(closeTimer);
Serial.print(" closeDiffTime:");
Serial.print(closeDiffTime);
Serial.print(" closeDiffTimeSec:");
Serial.print(closeDiffTimeSec);
Serial.print("\t");
Serial.println();

//Read the inputs 
s1_OpenReading = digitalRead(s1_OpenInput);
s1_CloseReading = digitalRead(s1_CloseInput);
s2_OpenReading = digitalRead(s2_OpenInput);
s2_CloseReading = digitalRead(s2_CloseInput);
//Debounce the inputs
s1_Open  = debounceAndDrive(s1_OpenInput,  s1_OpenOutput,  s1_Open,  lastS1_Open,  timerLastStateS1_Open,  debounceDelay);
s1_Close = debounceAndDrive(s1_CloseInput, s1_CloseOutput, s1_Close, lastS1_Close, timerLastStateS1_Close, debounceDelay);
s2_Open  = debounceAndDrive(s2_OpenInput,  s2_OpenOutput,  s2_Open,  lastS2_Open,  timerLastStateS2_Open,  debounceDelay);
s2_Close = debounceAndDrive(s2_CloseInput, s2_CloseOutput, s2_Close, lastS2_Close, timerLastStateS2_Close, debounceDelay);

//Update LED Status
digitalWrite(s1_OpenLED, digitalRead(s1_OpenInput));
digitalWrite(s1_CloseLED, digitalRead(s1_CloseInput));
digitalWrite(s2_OpenLED, digitalRead(s2_OpenInput));
digitalWrite(s2_CloseLED, digitalRead(s2_CloseInput));

//valve status update for use in LCD display
if (s1_Open == 0 && s1_Close==1){
  s1_Status='O';
} else if(s1_Open == 1 && s1_Close==0){
  s1_Status='C';
} else if(s1_Open == 1 && s1_Close==1){
  s1_Status='X';
}

if (s2_Open == 0 && s2_Close==1){
  s2_Status='O';
} else if(s2_Open == 1 && s2_Close==0){
  s2_Status='C';
} else if(s2_Open == 1 && s2_Close==1){
  s2_Status='X';
}


//Start the open timer
if(((s1_Open==0) || (s2_Open==0)) && openTimerStart==0 && (s1_Open != s2_Open) && fromClose==1){
  openTimer=millis();
  openTimerStart=1;
}
//stop the open timer
if(s1_Open==0 && s2_Open==0 && openTimerStart==1 && fromClose==1){
  openDiffTime=(float)(millis() - openTimer);
  openTimerStart=0;
  fromOpen=1;
  fromClose=0;
  closeDiffTime=0;
}

//start the close timer
if(((s1_Close==0) || (s2_Close==0)) && closeTimerStart==0 && (s1_Close != s2_Close) && fromOpen ==1){
  closeTimer=millis();
  closeTimerStart=1;
}

//stop the close timer
if (s1_Close==0 && s2_Close==0 && closeTimerStart==1 && fromOpen==1){
  closeDiffTime=(float)(millis() - closeTimer);
  closeTimerStart=0;
  fromClose=1;
  fromOpen=0;
  openDiffTime=0;
}

//Convert to seconds
openDiffTimeSec = openDiffTime/1000;
closeDiffTimeSec = closeDiffTime/1000;

lcdUpdate();                         // update LCD

}

void lcdUpdate()//Update LCD every 250 milliseconds
{
 static unsigned long lcdUpdateTime = millis();
 
 if ((millis()-lcdUpdateTime) > 250)
  {
   lcd.clear(); 
   lcd.print("S1:");
   lcd.print(s1_Status);
   lcd.print(" S2:");
   lcd.print(s2_Status);
   lcd.setCursor(0, 1); //starts 2nd line of text on LCD
   lcd.print("OPN:");
   lcd.print(openDiffTimeSec, 2);
   lcd.setCursor(8, 1);
   lcd.print("CLS:");
   lcd.print(closeDiffTimeSec, 2);
   lcd.print(digitalRead(s2_CloseInput));
   lcdUpdateTime = millis();
  }
 
}

// Debounce input and update its output pin
int debounceAndDrive(int inputPin, int outputPin, int &stableState, int &lastRaw, unsigned long &lastTime, unsigned long debounceDelay) {
  int raw = digitalRead(inputPin);

  // If input changed, reset timer
  if (raw != lastRaw) {
    lastTime = millis();
    lastRaw = raw;
  }

  // If stable for long enough, update stableState
  if ((millis() - lastTime) > debounceDelay) {
    if (raw != stableState) {
      stableState = raw;
      // Drive output accordingly (active low input, active low output)
      if (stableState == HIGH) {
        digitalWrite(outputPin, HIGH); // LED/indicator off
      } else {
        digitalWrite(outputPin, LOW);  // LED/indicator on
      }
    }
  }

  return stableState; // return the debounced stable state
}
